/*
 * Splits a single "vmtrace" log file into per-<bdev,inode> files.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include "list.h"
#define BITS_PER_LONG 32
#include "hash.h"
#include "vmtrace.h"


/* identify an open <bdev,inode> output file */
struct bdev_and_inode {
	unsigned long inode;
	unsigned long bdev;
	int open_fd;
	struct list_head hash;
	char *filename;
};

char *output_dir;

struct list_head *hash_table;
int hash_bits = 12;	/* 4096 hash buckets */

int init_hash_table(void)
{
	int i;
	int hash_entries = 1 << hash_bits;

	hash_table = (struct list_head *) malloc(sizeof(struct list_head) *hash_entries);

	printf("hash_size: %d bytes\n", sizeof(struct list_head) * hash_entries);

	if (!hash_table) {
		printf("failed to malloc hash table!\n");
		return -1;
	}
	for (i=0;i<hash_entries;i++)
		INIT_LIST_HEAD(&hash_table[i]);

	return 0;
}

int insert_into_cache(struct bdev_and_inode *bdevino)
{
	int bucket = hash_long(bdevino->inode, hash_bits);
	list_add(&bdevino->hash, &hash_table[bucket]);
}

struct bdev_and_inode* lookup_into_cache (unsigned long inode,
		unsigned long bdev)
{
	struct list_head *entry;
	int bucket = hash_long(inode, hash_bits);

	list_for_each(entry, &hash_table[bucket]) {
		struct bdev_and_inode *bdevino;
		bdevino = list_entry(entry, struct bdev_and_inode, hash);
		if (bdevino->bdev == bdev)
			return bdevino;
	}
	return NULL;
}

struct bdev_and_inode *create_openfile(unsigned long inode, unsigned long bdev)
{
	struct bdev_and_inode *bdevino;
	int fd;
	char pathname[64];

	snprintf(pathname, sizeof(pathname), "%x-%d", inode, bdev);

	printf("%s\n", pathname);

	if ((fd = open(pathname, O_RDWR|O_CREAT)) < 0) {
		perror("open");
		printf("pathname: %s\n", pathname);
	}

	bdevino = (struct bdev_and_inode*)malloc(sizeof(struct bdev_and_inode));

	if (!bdevino) {
		printf("bdevino allocation failure!");
		exit(0);
	}

	bdevino->open_fd = fd;
	bdevino->inode = inode;
	bdevino->bdev = bdev;
	INIT_LIST_HEAD(&bdevino->hash);
	bdevino->filename = (char *)malloc(strlen(pathname));
	if (!bdevino->filename) {
		printf("bdevino filename allocation failure!\n");
		exit(0);
	}
	strcpy(bdevino->filename, pathname);
	insert_into_cache(bdevino);
	return bdevino;
}

int write_entry(struct bdev_and_inode *bdevino, struct vm_trace_entry *entry)
{
	int ret;

	ret = write(bdevino->open_fd, entry, sizeof(struct vm_trace_entry));

	if (ret < 0) {
		printf("error writing to %s\n", bdevino->filename);
		perror("write");
		return 0;
	}
	return 1;
}

int parse_trace_file(void *mptr, int len)
{
	int i;
	struct vm_trace_entry *entry = (struct vm_trace_entry *) mptr;

	for(i = 0; i < len/sizeof(struct vm_trace_entry); i++) {
		struct bdev_and_inode *bdevino;

		bdevino = lookup_into_cache(entry->inode, entry->bdev);
		if (!bdevino)
			bdevino = create_openfile(entry->inode, entry->bdev);
		if (!bdevino) {
			printf("failure to create output bdevinode file!\n");
			exit(0);
		}

		if (!write_entry(bdevino, entry))
			exit(0);
		entry++;
	}
}


int create_dir(char *dir)
{
	int fd, ret, serrno;

	ret = mkdir(dir, S_IXUSR|S_IWUSR|S_IRUSR);

	serrno = errno;

	if (ret < 0 && serrno != EEXIST) {
		perror("mkdir");
		return -1;
	}

	output_dir = (char *)malloc(strlen(dir));
	if (!output_dir) {
		printf("create_dir malloc failure!\n");
		return -1;
	}

	strcpy(output_dir, dir);

	printf("output_dir:%s\n", output_dir);
}

int main(int argc, char *argv[])
{
	int fd;
	void *mptr;
	struct stat fstat;

	if (argc != 3) {
		printf("usage: vmtrace-split trace-file directory\n");
		exit(0);
	}

	if (stat(argv[1], &fstat) < 0) {
		perror("stat");
		exit(0);
	}

	if ((fd = open(argv[1], O_RDONLY)) < 0) {
		perror("open");
		exit(0);
	}

	mptr = mmap(0, fstat.st_size, PROT_READ, MAP_SHARED, fd, 0);

	if (mptr == MAP_FAILED) {
		perror("mmap");
		exit(0);
	}

	if (create_dir(argv[2]) < 0) {
		printf("creation of directory failed!\n");
		exit(0);
	}

	if (init_hash_table() < 0) {
		printf("creation of hashtable failed!\n");
		exit(0);
	}

	if (chdir(argv[2]) < 0) {
		perror("chdir");
		exit(0);
	}

	parse_trace_file(mptr, fstat.st_size);

	munmap(mptr, fstat.st_size);
	close(fd);
}

