/*
 * Calculate the numerical relation between accesses to two different
 * mappings. This is an attempt to estimate how interleaved the accesses
 * are.
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#define BITS_PER_LONG 32
#include "list.h"
#include "vmtrace.h"

/* 
 * each access_tuple represents the continuous access of 
 * the application to the "A" mapping, and another 
 * continuous access to the "B" mapping.
 * 
 * By recording this information we can estimate how interleaved
 * accesses are between each mapping.
 */
struct access_tuple {
	unsigned int a_nraccess;
	unsigned int b_nraccess;
	struct list_head list;
};

LIST_HEAD(tuple_list);

int nr_tuples = 0;
int nr_accesses = 0;

struct access_tuple *alloc_tuple(void)
{
	struct access_tuple *tuple;
	
	tuple = malloc(sizeof(struct access_tuple));
	if (!tuple) {
		printf("tuple allocation failed!\n");
		exit(0);
	}
	tuple->a_nraccess = 0;
	tuple->b_nraccess = 0;
	INIT_LIST_HEAD(&tuple->list);
	return tuple;
}

int parse_trace_file(void *mptr, int len, unsigned long inoa,
			unsigned long inob)
{
	int i;
	unsigned long cur_mapping = -1;
	struct access_tuple *cur_tuple = NULL;
	struct vm_trace_entry *entry = (struct vm_trace_entry *) mptr;

	for(i = 0; i < len/sizeof(struct vm_trace_entry); i++) {
		if (entry->inode == inoa) {
			if (cur_mapping == -1) {
				cur_mapping = inoa;
				cur_tuple = alloc_tuple();
			}

			cur_tuple->a_nraccess++;
			if (cur_mapping != inoa) {
				list_add_tail(&cur_tuple->list, &tuple_list);
				cur_tuple = alloc_tuple();
				nr_tuples++;
				cur_mapping = inoa;
			}
			nr_accesses++;
		} else if (entry->inode == inob) {
			if (cur_mapping == -1) {
				cur_mapping = inob;
				cur_tuple = alloc_tuple();
			}

			cur_tuple->b_nraccess++;
			if (cur_mapping != inob) {
				list_add_tail(&cur_tuple->list, &tuple_list);
				cur_tuple = alloc_tuple();
				nr_tuples++;
				cur_mapping = inob;
			}
			nr_accesses++;
		}
			
		entry++;
	}
}

void dump_stats(void)
{
	struct access_tuple *walk;
	struct list_head *entry;
	unsigned long nr_a, nr_b, nr_t;

	nr_a = nr_b = nr_t = 0;

	printf("interleave rate: %f\n", (float)(nr_tuples)/
					(float)(nr_accesses));
	printf("0.5 = full interleave (A/B/A/B/A/B...)\n");
	printf("0 = no interleaving\n");

	list_for_each(entry, &tuple_list) {
		walk = list_entry(entry, struct access_tuple, list);
		nr_a += walk->a_nraccess;
		nr_b += walk->b_nraccess;
		nr_t++;
	}
	printf("nr_tuples:%d\n", nr_tuples);
	printf("nr_accesses:%d\n", nr_accesses);
	printf("nr_a: %d\n", nr_a);
	printf("nr_b: %d\n", nr_b);
	printf("nr_t: %d\n", nr_t);
	printf("avg. rate between accesses of:\n");
	printf(" A	  %f\n", (float)nr_a/(float)nr_tuples);
	printf("---	 ----------\n");
	printf(" B	  %f\n", (float)nr_b/(float)nr_tuples);
}


int main(int argc, char *argv[]) 
{
	int fd;
	void *mptr;
	struct stat fstat;
	unsigned long inonr_a, inonr_b;

	if (argc != 4) {
		printf("usage: vmtrace-relation trace-file inonr_a inonr_b\n");
		exit(0);
	}

	if (stat(argv[1], &fstat) < 0) {
		perror("stat");
		exit(0);
	}

	if ((fd = open(argv[1], O_RDONLY)) < 0) {
		perror("open");
		exit(0);
	}

	mptr = mmap(0, fstat.st_size, PROT_READ, MAP_SHARED, fd, 0);

	if (mptr == MAP_FAILED) {
		perror("mmap");	
		exit(0);
	}

	inonr_a = strtoul(argv[2], NULL, 16);
	inonr_b = strtoul(argv[3], NULL, 16);

	printf("comparing inonr_a:%x inonr_b:%x\n", inonr_a, inonr_b);
	
	parse_trace_file(mptr, fstat.st_size, inonr_a, inonr_b);

	dump_stats();

	munmap(mptr, fstat.st_size);
	close(fd);
}

