/*
 * Calculate avg. Inter Reference Period of all pages in a file (and
 * avg of all pages IRP's).
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include "list.h"
#define BITS_PER_LONG 32
#include "hash.h"
#include "vmtrace.h"

#define PAGE_REACCESSED 0x1

/* identify a page */
struct page {
	unsigned long offset;
	unsigned long flags;
	unsigned long long last_access;
	double inv_tot_delta;
	unsigned long long max_delta;
	unsigned long long min_delta;
	unsigned int nr_accesses;
	struct list_head hash;
	unsigned long last_seqnr;
};

#define MHZ 1133

unsigned long cycles_per_ns = 0;
unsigned long calc_cycles_per_ns(unsigned int mhz)
{
        unsigned long long hz = mhz * 1024 * 1024;
        if (hz > 100000000)
                return hz/100000000; /* divide by 100 million */
        /* does not support less than 100MHz */
}

unsigned long long cycles_to_ns(unsigned long long cycles)
{
	return cycles/cycles_per_ns;
}

struct list_head *hash_table;
int hash_bits = 12;	/* 4096 hash buckets */

int init_hash_table(void)
{
	int i;
	int hash_entries = 1 << hash_bits;

	hash_table = (struct list_head *) malloc(sizeof(struct list_head) *hash_entries);

	printf("hash_size: %d bytes\n", sizeof(struct list_head) * hash_entries);

	if (!hash_table) {
		printf("failed to malloc hash table!\n");
		return -1;
	}
	for (i=0;i<hash_entries;i++)
		INIT_LIST_HEAD(&hash_table[i]);

	return 0;
}

int insert_into_cache(struct page *page)
{
	int bucket = hash_long(page->offset, hash_bits);
	list_add(&page->hash, &hash_table[bucket]);
}

struct page* lookup_into_cache (unsigned long offset)
{
	struct list_head *entry;
	int bucket = hash_long(offset, hash_bits);

	list_for_each(entry, &hash_table[bucket]) {
		struct page *page;
		page = list_entry(entry, struct page, hash);
		if (page->offset == offset) {
			return page;
		}
	}
	return NULL;
}

struct page *create_page(unsigned long offset)
{
	struct page *page;

	page = (struct page*)malloc(sizeof(struct page));

	if (!page) {
		printf("struct page allocation failure!");
		exit(0);
	}

	page->offset = offset;
	page->flags = 0;
	page->last_access = 0;
	page->last_seqnr = 0;
	page->nr_accesses = 0;
	page->inv_tot_delta = 0;
	page->max_delta = 0;
	page->min_delta = -1;
	INIT_LIST_HEAD(&page->hash);
	insert_into_cache(page);

	return page;
}

int calculate_page_data(struct page *page, struct vm_trace_entry *entry)
{
	unsigned long long tstamp = entry->tstamp;
	unsigned long long delta = tstamp - page->last_access;

	if (page->last_access == 0)
		printf("page last access is zero!\n");

	if (tstamp - page->last_access > tstamp) {
		printf("OVERFLOW! tstamp:%llx page->last_access:%llx\n",
			tstamp, page->last_access);
	}

	if (delta > page->max_delta)
		page->max_delta = delta;
	if (delta < page->min_delta)
		page->min_delta = delta;

	page->inv_tot_delta += (1000 * (double)1 / (double)(delta));

	page->last_seqnr = entry->seqnr;

	page->nr_accesses++;

	page->flags |= PAGE_REACCESSED;
}

int parse_trace_file(void *mptr, int len)
{
	int i;
	struct vm_trace_entry *entry = (struct vm_trace_entry *) mptr;

	for(i = 0; i < len/sizeof(struct vm_trace_entry); i++) {
		struct page *page;
		unsigned long offset = entry->offset / (1 << 12);
		page = lookup_into_cache(offset);
		if (!page)
			page = create_page(offset);
		if (!page) {
			printf("failure to create struct page!\n");
			exit(0);
		}

		if (page->last_access != 0)
			calculate_page_data(page, entry);

		page->last_access = entry->tstamp;

		entry++;
	}
}

void walk_hash_table(char *filename)
{
	int i, nrpages = 0;
	int hash_entries = 1 << hash_bits;
	struct list_head *head;
	double inv_total_delta;

	for (i=0; i<hash_entries; i++) {
		struct page *page;
		struct list_head *entry;

		head = &hash_table[i];
		list_for_each(entry, head) {
			double avg_irf;
			page = list_entry(entry, struct page, hash);
			if (!(page->flags & PAGE_REACCESSED))
				continue;

			avg_irf = (double)page->inv_tot_delta /
				  (double)page->nr_accesses;

			inv_total_delta += avg_irf;
			nrpages++;
#if 0
			printf("inv_tot_delta:%f nr_accesses:%d\n",
				page->inv_tot_delta, page->nr_accesses);

			printf("i:%x avg.irf:%f maxdelt:%ld mindelt:%ld\n",
				page->offset, avg_irf, page->max_delta,
				page->min_delta);
#endif
		}
	}
	if (nrpages)
		printf("%s: avg. IRF of all reaccessed pages(%d): %f\n", filename,
			       	nrpages, inv_total_delta/(double)nrpages);
}


int main(int argc, char *argv[])
{
	int fd;
	void *mptr;
	struct stat fstat;

	if (argc != 2) {
		printf("usage: vmtrace-irp file-trace\n");
		exit(0);
	}

	if (stat(argv[1], &fstat) < 0) {
		perror("stat");
		exit(0);
	}

	if ((fd = open(argv[1], O_RDONLY)) < 0) {
		perror("open");
		exit(0);
	}

	mptr = mmap(0, fstat.st_size, PROT_READ, MAP_SHARED, fd, 0);

	if (mptr == MAP_FAILED) {
		perror("mmap");
		exit(0);
	}

	if (init_hash_table() < 0) {
		printf("creation of hashtable failed!\n");
		exit(0);
	}

	cycles_per_ns = calc_cycles_per_ns(MHZ);

	parse_trace_file(mptr, fstat.st_size);

	walk_hash_table(argv[1]);

	munmap(mptr, fstat.st_size);
	close(fd);
}

